/*************************************************************************
 *
 * Hitchhiker's Guide to the IBM PE 
 * Ray trace program with bug corrected
 * Chapter 3 - Don't Panic
 *
 * To compile:
 * mpcc -g -o rtrace_good rtrace_good.c
 *
 *
 * Description: 
 * This is a sample program that partitions N tasks into
 * two groups, a collect node and N - 1 compute nodes.
 * The responsibility of the collect node is to collect the data
 * generated by the compute nodes. The compute nodes send the 
 * results of their work to the collect node for collection.
 *
 * The bug in the original code was due to the fact that each processing
 * task determined the rows to cover by dividing the total number of
 * rows by the number of processing tasks.  If that division was not
 * integral, the number of pixels processed was less than the number of
 * pixels expected by the collection task and that task waited 
 * indefinitely for more input.
 *
 * The solution is to allocate the pixels among the processing tasks
 * in such a manner as to ensure that all pixels are processed.
 *
 *******************************************************************/

#include <mpi.h>

#define PIXEL_WIDTH	50
#define PIXEL_HEIGHT	50

int	First_Line = 0;

void main(int argc, char *argv[])
{ 
  int    	numtask;
  int	taskid;
 
  /* Find out number of tasks/nodes. */
  MPI_Init( &argc, &argv);
  MPI_Comm_size( MPI_COMM_WORLD, &numtask);
  MPI_Comm_rank( MPI_COMM_WORLD, &taskid);
 
  /* Task 0 is the coordinator and collects the processed pixels */
  /* All the other tasks process the pixels                      */
  if ( taskid == 0 )
    collect_pixels(taskid, numtask);
  else
    compute_pixels(taskid, numtask);

  printf("Task %d waiting to complete.\n", taskid);
  /* Wait for everybody to complete */
  MPI_Barrier(MPI_COMM_WORLD);
  printf("Task %d complete.\n",taskid);
  MPI_Finalize();
  exit(); 
}

/* In a real implementation, this routine would process the pixel */
/* in some manner and send back the processed pixel along with its*/
/* location.  Since we're not processing the pixel, all we do is  */
/* send back the location                                         */
compute_pixels(int taskid, int numtask)
{
  int		offset;
  int		row, col;
  int		pixel_data[2];
  MPI_Status	stat;

  printf("Compute #%d: checking in\n", taskid);
	
  First_Line = (taskid - 1);            /* First n-1 rows are assigned */
					/* to processing tasks         */
  offset = numtask - 1;                 /* Each task skips over rows   */
					/* processed by other tasks    */

  /* Go through entire pixel buffer, jumping ahead by numtask-1 each time */
  for (row = First_Line; row < PIXEL_HEIGHT; row += offset)
    for ( col = 0; col < PIXEL_WIDTH; col ++)
      {
	pixel_data[0] = row;
	pixel_data[1] = col;
	MPI_Send(pixel_data, 2, MPI_INT, 0, 0, MPI_COMM_WORLD);
      }
  printf("Compute #%d: done sending. ", taskid);
  return;
}

/* This routine collects the pixels.  In a real implementation, */
/* after receiving the pixel data, the routine would look at the*/
/* location information that came back with the pixel and move  */
/* the pixel into the appropriate place in the working buffer   */
/* Since we aren't doing anything with the pixel data, we don't */
/* bother and each message overwrites the previous one          */
collect_pixels(int taskid, int numtask)
{
  int		pixel_data[2];
  MPI_Status	stat;
  int     	mx = PIXEL_HEIGHT * PIXEL_WIDTH;

  printf("Control #%d: No. of nodes used is %d\n",taskid,numtask);
  printf("Control: expect to receive %d messages\n", mx);

  while (mx > 0)
    {
      MPI_Recv(pixel_data, 2, MPI_INT, MPI_ANY_SOURCE, 
	       MPI_ANY_TAG, MPI_COMM_WORLD, &stat);
      mx--;
    }
  printf("Control node #%d: done receiving. ",taskid);
  return;
}
